define([
    'backbone',
    'User',
    'modules/new-appointment-request/resources/common/model',
], function(Backbone, user, Model) {
    'use strict';

    describe('Base Model for All Scheduling', function() {
        var model;

        beforeEach(function() {
            model = new Model();
        });

        describe('Abstract functions', function() {
            var ABSTRACT_MESSAGE = 'This function is intended to be overridden by extending model';

            it('throws on error on save', function() {
                expect(function() {
                    model.save();
                }).toThrowError(Error, ABSTRACT_MESSAGE);
            });

            it('throws on error on getUrl', function() {
                expect(function() {
                    model.getUrl();
                }).toThrowError(Error, ABSTRACT_MESSAGE);
            });

            it('throws an error on createSaveData', function() {
                expect(function() {
                    model.createSaveData();
                }).toThrowError(Error, ABSTRACT_MESSAGE);
            });
        });

        describe('Facility Wrapper', function() {
            describe('With Facility', function() {
                var FACILITY = {
                    name: 'Scared Heart',
                    institutionCode: '123AB',
                    rootStationCode: '123',
                };
                var facility;

                beforeEach(function() {
                    facility = new Backbone.Model(FACILITY);
                    model.set('facility', facility);
                });

                it('returns the facility model', function() {
                    expect(model.facility()).toBe(facility);
                });

                it('returns the correct facility name', function() {
                    expect(model.getFacilityName()).toBe(FACILITY.name);
                });

                it('returns the correct institution code', function() {
                    expect(model.getInstitutionCode()).toBe(FACILITY.institutionCode);
                });

                it('returns the correct site code', function() {
                    expect(model.getSiteCode()).toBe(FACILITY.rootStationCode);
                });
            });

            describe('Without Facility', function() {
                var NONE = '';

                it('returns an empty string when there is no facilityName', function() {
                    expect(model.getFacilityName()).toBe(NONE);
                });

                it('returns an empty string when there is no institution code', function() {
                    expect(model.getInstitutionCode()).toBe(NONE);
                });

                it('returns an empty string when there is no site code', function() {
                    expect(model.getSiteCode()).toBe(NONE);
                });
            });
        });

        describe('User Wrapper', function() {
            it('gets the current user', function() {
                expect(model.user()).toBe(user);
            });

            it('gets the correct user id', function() {
                expect(model.getUserId()).toBe(user.get('id'));
            });

            it('gets the correct id type for the user', function() {
                expect(model.getUserIdType()).toBe(user.get('idType'));
            });
        });

        describe('Type of Care Wrapper', function() {
            function createModel(id) {
                return new Backbone.Model({
                    id: id,
                    name: 'whatever',
                });
            }

            it('returns the correct type of care', function() {
                var toc = createModel('123');
                model.set('typeOfCare', toc);
                expect(model.typeOfCare()).toBe(toc);
            });

            it('returns true when type of care is primary', function() {
                model.set('typeOfCare', createModel('323'));
                expect(model.isPrimaryCare()).toBeTruthy();
            });

            it('returns false when type of care is not primary', function() {
                model.set('typeOfCare', createModel('000'));
                expect(model.isPrimaryCare()).toBeFalsy();
            });

            it('returns true when type of care is express', function() {
                model.set('typeOfCare', createModel('CR1'));
                expect(model.isExpressCare()).toBeTruthy();
            });

            it('returns false when type of care is not express', function() {
                model.set('typeOfCare', createModel('000'));
                expect(model.isExpressCare()).toBeFalsy();
            });

            it('returns empty string with type of care is not set', function() {
                expect(model.getCareName()).toBe('');
            });

            it('returns the name when typeof care is set', function() {
                model.set('typeOfCare', createModel('000'));
                expect(model.getCareName()).toBe('whatever');
            });
        });

        describe('Email Wrapper', function() {
            it('returns an empty object when email is not allowed', function() {
                expect(model.getEmail()).toEqual({});
            });

            it('returns a simple email object when email is allowed', function() {
                var email = new Backbone.Model({
                    emailAllowed: true,
                    emailAddress: 'test@test.com',
                });
                var results;

                model.set('emailPreferences', email);

                results = model.getEmail();

                expect(results.emailAllowed).toBeTruthy();
                expect(results.email).toBe(email.get('emailAddress'));
            });
        });

        describe('Scheduling Type', function() {
            it('validates request scheduling correctly', function() {
                model.set('scheduling-method', 'clerk');
                expect(model.isRequest()).toBeTruthy();
                expect(model.isDirect()).toBeFalsy();
            });

            it('validates direct scheduling correctly', function() {
                model.set('scheduling-method', 'direct');
                expect(model.isRequest()).toBeFalsy();
                expect(model.isDirect()).toBeTruthy();
            });
        });
    });
});
